<?php

class FileIPList {
    private $fileinfo;

    public function __construct($fileinfo) {
        $this->fileinfo = $fileinfo;
    }

    public function get_ip_list() {
        if (!$this->fileinfo->isFile()){
            throw new ExceptionRUMError("IP file not found.");
        }

        $ip_file = $this->fileinfo->openFile('r');
        $content = '';
        while (!$ip_file->eof()) {
            $content .= $ip_file->fgets();
        }

        return $content;
    }

    public function is_ip_allowed($ip) {
        $remote_ip_version = $this->get_ip_version($ip);

        if (!$remote_ip_version){ throw new ExceptionRUMError("invalid remote ip address '$ip'"); }

        $ip_file = $this->fileinfo->openFile('r');

        while (!$ip_file->eof()){
            $line = trim($ip_file->fgets());
            if (!$line) { continue; }
            $parts = explode('/', $line, 2);
            $range_ip = $parts[0];
            $range_bits = isset($parts[1]) ? $parts[1] : NULL;

            $range_ip_version = $this->get_ip_version($range_ip);
            if (!$range_ip_version){
                throw new ExceptionRUMError("invalid ip address range in IP file: $line");
            }

            if ($range_ip_version != $remote_ip_version){ continue; }

            if ($this->is_ip_in_range($ip, $range_ip, $range_bits)){
                return 1;
            }
        }

        $ip_file = null;

        return false;
    }

    private function get_ip_version($ip){
        if(filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV4)){
            return 4;
        }
        if(filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6)){
            return 6;
        }
        return 0;
    }

    private function is_ip_in_range($ip, $range_ip, $range_bits){
        if (!isset($range_bits) && ($ip === $range_ip)) { return true; }

        $binaryip = $this->ip_to_bits($ip);
        $binarynet = $this->ip_to_bits($range_ip);

        $ip_net_bits = substr($binaryip, 0, $range_bits);
        $net_bits = substr($binarynet, 0, $range_bits);

        return ($ip_net_bits === $net_bits);
    }

    private function ip_to_bits($ip)
    {
        $inet = inet_pton($ip);
        $unpacked = unpack('A'.strlen($inet), $inet);
        $unpacked = str_split($unpacked[1]);
        $binaryip = '';
        foreach ($unpacked as $char) {
            $binaryip .= str_pad(decbin(ord($char)), 8, '0', STR_PAD_LEFT);
        }
        return $binaryip;
    }
}
