<?php

class Client {
    private $logger;
    private $process_by;
    private $ip_list;

    private $TRN_TYPE_PARAMS =
        array(
            'add' => array('usercode', 'passcode', 'amount'),
            'modify' => array('usercode', 'passcode'),
            'rebill' => array('usercode', 'amount', 'trn_id'),
            'cancel' => array('usercode'),
            'delete' => array('usercode'),
            'list_ip' => array()
        );
    private $PARAM_VALIDATOR = array(
        'usercode' => '/^[\w_][\w_ ]{0,11}$/',
        'passcode' => '/^.{5}.*$/',
        'amount'   => '/^\d{1,3}(\.\d{0,2})?$/',
        'trn_id'   => '/^\d{1,12}$/'
    );

    public function __construct($logger, $process_by, $ip_list) {
        $this->logger = $logger;
        $this->process_by = $process_by;
        $this->ip_list = $ip_list;
    }

    public function process($params) {
        $this->log_request($params);

        try {
            $this->check_remote_ip_allowed($_SERVER['REMOTE_ADDR']);
            $trn_type = $this->get_trn_type($params);
            $this->validate_params($trn_type, $params);
            return $this->dispatch_trn_type($trn_type, $params);
        } catch (ExceptionRUMDecline $e) {
            $this->logger->warn($e->getMessage());
            return 'DECLINED';
        } catch (ExceptionRUMError $e) {
            $this->logger->error($e->getMessage());
            return "ERROR\n".$e->getMessage();
        }
    }

    private function log_request($params){
        $message = 'connection from ';
        $message .= $_SERVER['REMOTE_ADDR'];

        foreach( $params as $key => $value ){
            if ($key == 'passcode') { continue; };
            $message .= ", $key '$value'";
        }

        $this->logger->info($message);
    }

    private function check_remote_ip_allowed($ip){
        if (!$this->ip_list->is_ip_allowed($_SERVER['REMOTE_ADDR'])){
            throw new ExceptionRUMError(
                "connection from ".$_SERVER['REMOTE_ADDR']." denied");
        }
    }

    private function get_trn_type($params){
        if (!isset($params['trn'])){
            throw new ExceptionRUMError("missing transaction type");
        }

        if (!isset($this->TRN_TYPE_PARAMS[$params['trn']])){
            throw new ExceptionRUMError("invalid transaction type ".$params['trn']);
        }

        return $params['trn'];
    }

    private function validate_params($trn_type, $params){
        foreach ($this->TRN_TYPE_PARAMS[$trn_type] as $param_name){
            if (!isset($params[$param_name])){
                throw new ExceptionRUMError("missing parameter $param_name");
            }
            if (!preg_match($this->PARAM_VALIDATOR[$param_name], $params[$param_name])){
                throw new ExceptionRUMError(
                    "invalid parameter $param_name '".$params[$param_name]."'");
            }
        }
    }

    private function dispatch_trn_type($trn_type, $params){
        switch($trn_type){
            case 'list_ip':
                return $this->ip_list->get_ip_list();
                break;
            case 'add':
                $this->process_by->add(
                    $params['usercode'], $params['passcode'], $params['amount']);
                $this->logger->info("user '".$params['usercode']."' added");
                break;
            case 'modify':
                $this->process_by->modify($params['usercode'], $params['passcode']);
                $this->logger->info("user '".$params['usercode']."' updated");
                break;
            case 'delete':
                $this->process_by->delete($params['usercode']);
                $this->logger->info("user '".$params['usercode']."' deleted");
                break;
            case 'rebill':
                $this->process_by->rebill(
                    $params['usercode'], $params['amount'], $params['trn_id']);
                break;
            case 'cancel':
                $this->process_by->cancel($params['usercode']);
                break;
            default:
                throw new ExceptionRUMError(
                    "unimplemented transaction type $trn_type");
        }
        return 'APPROVED';
    }
}
